/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#include "NeuralTanksLoadingScreen.h"
#include <Internationalization/StringTableRegistry.h>
#include <NeuralTanks/Utils/NeuralTanksUtils.h>

/// <summary>
/// Called when this SLATE UI is first built.
/// </summary>
/// <param name="InArgs">The additional arguments (in our case, the LoadingScreenTip).</param>
void SNeuralTanksLoadingScreen::Construct(const FArguments& InArgs)
{
	FText localizedNowLoadingText;
	FSlateFontInfo neuralTanksFontInfo;
	UFont* neuralTanksFont;

	myLoadingScreenTip = InArgs._LoadingScreenTip.Get();

	neuralTanksFont = LoadObject<UFont>(NULL, TEXT("/Game/Slate/Fonts/NeuralTanksFont"));
	if (neuralTanksFont != NULL)
	{
		neuralTanksFontInfo = FSlateFontInfo(neuralTanksFont, 24);
		localizedNowLoadingText = FText::FromStringTable(TEXT("/Game/Text/NeuralTanksStringTable"), TEXT("LoadingScreen_NowLoading"));

		ChildSlot
			[
				SNew(SVerticalBox)
					+ SVerticalBox::Slot()
					.VAlign(VAlign_Center)
					.HAlign(HAlign_Center)
					[
						SNew(STextBlock)
							.Font(neuralTanksFontInfo)
							.Text(localizedNowLoadingText)
							.Visibility(this, &SNeuralTanksLoadingScreen::GetMessageIndicatorVisibility)
					]
					+ SVerticalBox::Slot()
					.VAlign(VAlign_Center)
					.HAlign(HAlign_Center)
					[
						SNew(SThrobber)
							.Visibility(this, &SNeuralTanksLoadingScreen::GetLoadIndicatorVisibility)
					]
					+ SVerticalBox::Slot()
					.VAlign(VAlign_Center)
					.HAlign(HAlign_Center)
					[
						SNew(STextBlock)
							.AutoWrapText(true)
							.Font(neuralTanksFontInfo)
							.Text(myLoadingScreenTip)
							.Visibility(this, &SNeuralTanksLoadingScreen::GetTipsVisibility)
					]
			];
	}
	else
	{
		UNeuralTanksUtils::WriteToLog("SNeuralTanksLoadingScreen::Construct(const FArguments&) Error: Failed to find NeuralTanksFont!");
	}

}

/// <summary>
/// Returns true if the loading indicator should be visible, false otherwise.
/// This is dependent on if loading is finished or not.
/// </summary>
/// <returns>true if the loading indicator should be visible, false otherwise.</returns>
EVisibility SNeuralTanksLoadingScreen::GetLoadIndicatorVisibility() const
{
	return GetMoviePlayer()->IsLoadingFinished() ? EVisibility::Collapsed : EVisibility::Visible;
}

/// <summary>
/// Returns true if the message indicator should be visible, false otherwise.
/// This is dependent on if loading is finished or not.
/// </summary>
/// <returns>true if the message indicator should be visible, false otherwise.</returns>
EVisibility SNeuralTanksLoadingScreen::GetMessageIndicatorVisibility() const
{
	return GetMoviePlayer()->IsLoadingFinished() ? EVisibility::Collapsed : EVisibility::Visible;
}

/// <summary>
/// Returns true if the localized tip should be visible, false otherwise.
/// This is dependent on if loading is finished or not.
/// </summary>
/// <returns>true if the localized tip should be visible, false otherwise.</returns>
EVisibility SNeuralTanksLoadingScreen::GetTipsVisibility() const
{
	return GetMoviePlayer()->IsLoadingFinished() ? EVisibility::Collapsed : EVisibility::Visible;
}
