/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#pragma once
#include "../Items/TankItem.h"
#include "TankItemDataCollection.generated.h"

/// <summary>
/// The <c>UTankItemDataCollection</c> class represents a collection of <c>UTankItemData</c>s.
/// This is used in the store for bundling items together. For example, when the player buys
/// a camouflage, we give them 5 accessories to equip (one for each equipment slot on the tank).
/// 
/// Without this, the player would need to individually buy each camouflage for each equipment slot
/// (5 different purchases). In addition to this, the collection also has a corresponding purchase cost,
/// localized description, store item texture, and the collection of items.
/// </summary>
UCLASS(Blueprintable, BlueprintType)
class UTankItemDataCollection : public UObject
{
	GENERATED_BODY()

public:
	/// <summary>
	/// The amount of money it costs for this collection (series of items) from the store.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	int CollectionCost;

	/// <summary>
	/// The localized collection name.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	FText CollectionName;

	/// <summary>
	/// The localized description.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	FText CollectionDescription;

	/// <summary>
	/// The image used to represent this collection in the store.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	UTexture2D* InventoryTexture;

	/// <summary>
	/// The items to give the player (add to their inventory) 
	/// after purchasing this collection.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	TArray<TSubclassOf<UTankItemData>> TankItemDatas;
};