/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#pragma once
#include "../Items/TankItem.h"
#include "TankItemData.generated.h"

/// <summary>
/// This class holds the data of a particular tank item. <c>ATankItem</c> is the actor
/// version of this class and exists in the game world. In the inventory, game save,
/// and in other cases, the item is converted to its UTankItemData equivalent so it
/// can be saved/loaded. 
/// </summary>
UCLASS(Blueprintable, BlueprintType)
class UTankItemData : public UObject
{
	GENERATED_BODY()

public:
	/// <summary>
	/// The amount of money this item costs to buy in the store.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	int ItemCost;

	/// <summary>
	/// The category type for this item (Ammo, Vanity, Equipment, etc).
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	EItemType ItemType;

	/// <summary>
	/// The specific slot on the tank this item fits into (Boddy, Cannon, Hitch, etc).
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	EEquipmentSlot EquipmentSlot;

	/// <summary>
	/// The name of this item (not localized since it is used for reference lookup).
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	FString ItemName;

	/// <summary>
	/// The description of this item (localized).
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	FText ItemDescription;

	/// <summary>
	/// The image used to represent this item in the inventory.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	UTexture2D* InventoryTexture;

	/// <summary>
	/// The static mesh to use (if available) for this item (the thing that is visually attached to the tank).
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	UStaticMesh* StaticMesh;

	/// <summary>
	/// The skeletal mesh to use (if available) for this item (the thing that is visually attached to the tank).
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	USkeletalMesh* SkeletalMesh;

	/// <summary>
	/// Additional static meshes to automatically attach when this item is attached. This is typically
	/// used for the tank turret because the LMGs and the LMG mounts are separate from the turret static mesh.
	/// </summary>
	UPROPERTY(EditAnywhere, SaveGame, BlueprintReadWrite)
	TMap<FString, UStaticMesh*> AdditionalStaticMeshes;

	/// <summary>
	/// The serialized byte array with the above data.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite)
	TArray<uint8> ByteData;
};