/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#pragma once
#include <Sound/SoundWave.h>
#include <Components/AudioComponent.h>
#include "AITankPawn.h"
#include "BossAITankPawn.generated.h"

/// <summary>
/// Boss AI Tank Pawn is a child of the AI Tank Pawn. This is used for bosses (such as Sassy Sam).
/// The Boss AI Tank Pawn has all the capabilities of AI Tank Pawn but additionally has support
/// for voice lines (and playing voice lines when hit).
/// </summary>
UCLASS(config = Game)
class ABossAITankPawn : public AAITankPawn
{
	GENERATED_BODY()

protected:
	/// <summary>
	/// True when this Boss AI Tank has been hit for the first time, false otherwise.
	/// This is used to play a particular voice line.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Internal")
	bool myHasBeenHitOnce;

	/// <summary>
	/// True if a voice line is currently playing, false otherwise.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Internal")
	bool myIsLinePlaying;

	/// <summary>
	/// The Audio Component used for playing voice lines and also tracking when
	/// they change state (stopped, pause, playing, etc).
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "Internal")
	UAudioComponent* myAudioComponent;

	/// <summary>
	/// The voice line to play when this Boss AI Tank is hit for the first time.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	USoundWave* myOnFirstHitVoiceLine;

	/// <summary>
	/// The collection of voice lines to randomly play when this Boss AI Tank is hit.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	TArray<USoundWave*> myRandomVoiceLines;

public:

	/// <summary>
	/// Called when this Boss AI Tank is first spawned into the world/level.
	/// </summary>
	virtual void BeginPlay() override;

	/// <summary>
	/// Called when this Boss AI Tank is hit with a projectile. 
	/// Handles damage and playing voice lines.
	/// </summary>
	/// <param name="theAttackingPawn">The Tank that shot this Boss AI Tank.</param>
	/// <param name="damage">The amount of damage caused by the shot.</param>
	virtual void OnTankHit(ATankPawn* theAttackingPawn, float damage) override;

	/// <summary>
	/// Called when the currently playing audio changes stat, such as being stopped,
	/// paused, or starts playing. This allows us to keep track of when a voice line
	/// is currently being played or not.
	/// </summary>
	/// <param name="thePlayState">The updated state for the audio.</param>
	UFUNCTION()
	void OnPlayStateChanged(EAudioComponentPlayState thePlayState);

	/// <summary>
	/// Stops all voice lines from playing. This is used when the Boss AI Tank
	/// dies and we transition to a cutscene/cinematic.
	/// </summary>
	UFUNCTION(BlueprintCallable)
	void StopAllVoiceLines();
};