/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#pragma once
#include <Engine/LevelScriptActor.h>
#include "NeuralTanksGameLevel.generated.h"

/// <summary>
/// This class represents a single map/level/scene in the game.
/// It contains data for the minimum and maximum bounds the AI
/// can travel.
/// </summary>
UCLASS(BlueprintType)
class ANeuralTanksGameLevel : public ALevelScriptActor
{
	GENERATED_BODY()

protected:
	/// <summary>
	/// The lowest possible bounds the enemy tanks can travel.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	FVector myMinAIBoundsPosition;

	/// <summary>
	/// The highest possible bounds the enemy tanks can travel.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	FVector myMaxAIBoundsPosition;

public:
	/// <summary>
	/// Default constructor.
	/// </summary>
	ANeuralTanksGameLevel();

	/// <summary>
	/// Parses out and returns the numerical level number from the level name.
	/// For example, if the level name is "Level1-ThePit_C_0", this function 
	/// will return "1".
	/// </summary>
	/// <returns>The parsed integer value of the level (from the level name).</returns>
	int GetLevelNumberFromLevelName();

	/// <summary>
	/// Returns the name of the map/scene/level.
	/// </summary>
	/// <returns>The name of the map/scene/level.</returns>
	FString GetLevelName();

	/// <summary>
	/// Returns the lowest possible bounds the enemy tanks can travel.
	/// </summary>
	/// <returns>The lowest possible bounds the enemy tanks can travel.</returns>
	FVector GetMinAIBoundsPosition();

	/// <summary>
	/// Returns the highest possible bounds the enemy tanks can travel.
	/// </summary>
	/// <returns>The highest possible bounds the enemy tanks can travel.</returns>
	FVector GetMaxAIBoundsPosition();

	/// <summary>
	/// Updates/changes the lowest possible bounds the enemy tanks can travel.
	/// </summary>
	/// <param name="minAIBoundsPosition">The new lowest possible bounds the enemy tanks can travel.</param>
	void SetMinAIBoundsPosition(const FVector& minAIBoundsPosition);

	/// <summary>
	/// Updates/changes the highest possible bounds the enemy tanks can travel.
	/// </summary>
	/// <param name="maxAIBoundsPosition">The new highest possible bounds the enemy tanks can travel.</param>
	void SetMaxAIBoundsPosition(const FVector& maxAIBoundsPosition);

};