/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#include "NeuralTanksGameLevel.h"
#include "NeuralTanksGameMode.h"
#include "Utils/NeuralTanksUtils.h"
#include <Kismet/GameplayStatics.h>

/// <summary>
/// Default constructor.
/// </summary>
ANeuralTanksGameLevel::ANeuralTanksGameLevel()
{

}

/// <summary>
/// Parses out and returns the numerical level number from the level name.
/// For example, if the level name is "Level1-ThePit_C_0", this function 
/// will return "1".
/// </summary>
/// <returns>The parsed integer value of the level (from the level name).</returns>
int ANeuralTanksGameLevel::GetLevelNumberFromLevelName()
{
	int indexOfDash = -1;
	FString levelNumStr;
	FString levelName = GetLevelName(); //Example: "Level1-ThePit_C_0"

	if (levelName.IsEmpty() == false)
	{
		//Separate the string at the dash. 
		if (levelName.FindChar('-', indexOfDash) == true)
		{
			//Parse out the number portion of the level name.
			levelNumStr = levelName.Mid(5, indexOfDash - 5);
			if (levelNumStr.IsEmpty() == false)
			{
				//Convert the string version of the parsed number to its integer equivalent. 
				return FCString::Atoi(*levelNumStr);
			}	
		}
	}
	
	UNeuralTanksUtils::WriteToLog(TEXT("ANeuralTanksGameLevel::GetLevelNumberFromLevelName() CRITICAL ERROR: Failed to parse level number from level name!"));
	return -1;
}

/// <summary>
/// Returns the name of the map/scene/level.
/// </summary>
/// <returns>The name of the map/scene/level.</returns>
FString ANeuralTanksGameLevel::GetLevelName()
{
	return GetName();
}

/// <summary>
/// Returns the lowest possible bounds the enemy tanks can travel.
/// </summary>
/// <returns>The lowest possible bounds the enemy tanks can travel.</returns>
FVector ANeuralTanksGameLevel::GetMinAIBoundsPosition()
{
	return myMinAIBoundsPosition;
}

/// <summary>
/// Returns the highest possible bounds the enemy tanks can travel.
/// </summary>
/// <returns>The highest possible bounds the enemy tanks can travel.</returns>
FVector ANeuralTanksGameLevel::GetMaxAIBoundsPosition()
{
	return myMaxAIBoundsPosition;
}

/// <summary>
/// Updates/changes the lowest possible bounds the enemy tanks can travel.
/// </summary>
/// <param name="minAIBoundsPosition">The new lowest possible bounds the enemy tanks can travel.</param>
void ANeuralTanksGameLevel::SetMinAIBoundsPosition(const FVector& minAIBoundsPosition)
{
	myMinAIBoundsPosition = minAIBoundsPosition;
}

/// <summary>
/// Updates/changes the highest possible bounds the enemy tanks can travel.
/// </summary>
/// <param name="maxAIBoundsPosition">The new highest possible bounds the enemy tanks can travel.</param>
void ANeuralTanksGameLevel::SetMaxAIBoundsPosition(const FVector& maxAIBoundsPosition)
{
	myMaxAIBoundsPosition = maxAIBoundsPosition;
}