/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#pragma once
#include "NeuralTanksCheatManager.generated.h"

/// <summary>
/// The cheats available in the game.
/// </summary>
UENUM(BlueprintType)
enum class ECheatName : uint8
{
	VE_GODMODE 				UMETA(DisplayName = "God Mode"),
	VE_ALLGOLDARMOR			UMETA(DisplayName = "All Gold Armor"),
	VE_ALLLEVELSUNLOCKED	UMETA(DisplayName = "All Levels Unlocked"),
	VE_INSTANTRELOAD		UMETA(DisplayName = "Instant Reload"),
	VE_HASMAXCASH			UMETA(DisplayName = "Has Max Cash")
};

DECLARE_DYNAMIC_MULTICAST_DELEGATE_TwoParams(FOnCheatToggledSignature, ECheatName, cheatName, bool, isEnabled);

/// <summary>
/// Responsible for managing all of the cheats in the game. This class handles
/// enabling/disabling specified cheats, checking if they are enabled, and event handling
/// for if/when a cheat is enabled/disabled.
/// </summary>
UCLASS()
class UNeuralTanksCheatManager : public UObject
{
	GENERATED_BODY()

protected:
	/// <summary>
	/// A map containing all the available cheats and if 
	/// they are enabled or disabled for the player.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	TMap<ECheatName, bool> myCheats;

public:
	/// <summary>
	/// Event that can be registered to know when a cheat is enabled or disabled.
	/// </summary>
	FOnCheatToggledSignature OnCheatToggled_Event;

	/// <summary>
	/// Default constructor.
	/// </summary>
	UNeuralTanksCheatManager();

	/// <summary>
	/// Returns true if the provided cheat is enabled, false if it is disabled.
	/// </summary>
	/// <param name="cheatName">The cheat to check.</param>
	/// <returns>true if the provided cheat is enabled, false if it is disabled.</returns>
	UFUNCTION(BlueprintCallable)
	bool IsCheatEnabled(const ECheatName cheatName);

	/// <summary>
	/// Enables/disables a provided cheat for the player. 
	/// </summary>
	/// <param name="cheatName">The cheat to enable or disable.</param>
	/// <param name="isEnabled">true to enable the cheat, false to disable it.</param>
	UFUNCTION(BlueprintCallable)
	void ToggleCheat(const ECheatName cheatName, const bool isEnabled);
};