/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#pragma once
#include "NeuronInput.generated.h"

/// <summary>
/// <b>NeuronInput:</b> This is a small struct for holding a value and its corresponding weight.
/// </summary>
USTRUCT(BlueprintType)
struct FNeuronInput
{
	GENERATED_BODY()

	/// <summary>
	/// The value of this particular NeuronInput.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	float Value;

	/// <summary>
	/// The corresponding weight for this particular NeuronInput.
	/// </summary>
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "General")
	float Weight;

	/// <summary>
	/// Default constructor. Initializes Value and Weight to 0.
	/// </summary>
	FNeuronInput()
	{
		Value = 0.0;
		Weight = 0.0;
	}

	/// <summary>
	/// Parameterized constructor.
	/// </summary>
	/// <param name="inputValue">The value to assign to this NeuronInput's value.</param>
	/// <param name="inputWeight">The corresponding weight to assign to this NeuronInput's weight.</param>
	FNeuronInput(const float inputValue, const float inputWeight)
	{
		Value = inputValue;
		Weight = inputWeight;
	}

	/// <summary>
	/// Calculates the activation product of this particular NeuronInput by multiplying the value and weight.
	/// </summary>
	/// <returns></returns>
	float CalculateActivation()
	{
		return Value * Weight;
	}
};