/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#include "TankAmmunitionItem.h"

/// <summary>
/// Default constructor for the Tank Ammunition Item.
/// </summary>
ATankAmmunitionItem::ATankAmmunitionItem() : Super()
{
	myMinXPosition = 0.0f;
	myMinYPosition = 0.0f;
	myMaxXPosition = 0.0f;
	myMaxYPosition = 0.0f;
	myRespawnDropHeight = 2000.0f;
	myPositionBuffer = 50.0f;
}

/// <summary>
/// Initializes this Tank Ammunition Item with the provided data.
/// </summary>
/// <param name="itemCost">The ammount of money the item costs in the shop. (TankAmmunitionItem is not available in the shop, only the world/level.)</param>
/// <param name="itemType">The category of item such as Armor, Equipment, etc.</param>
/// <param name="itemName">The unique name for this item.</param>
/// <param name="itemDescription">The localized description of the name.</param>
/// <param name="minXLocation">The lower X bounds of where this item can exist in the world.</param>
/// <param name="minYLocation">The lower Y bounds of where this item can exist in the world.</param>
/// <param name="maxXLocation">The upper X bounds of where this item can exist in the world.</param>
/// <param name="maxYLocation">The upper Y bounds of where this item can exist in the world.</param>
void ATankAmmunitionItem::Initialize(const int itemCost, const EItemType itemType, const FString& itemName, const FText& itemDescription, const float& minXLocation, const float& minYLocation, const float& maxXLocation, const float& maxYLocation)
{
	Super::Initialize(itemCost, itemType, EEquipmentSlot::VE_NONE, itemName, itemDescription);

	myMinXPosition = minXLocation + myPositionBuffer;
	myMinYPosition = minYLocation + myPositionBuffer;
	myMaxXPosition = maxXLocation - myPositionBuffer;
	myMaxYPosition = maxYLocation - myPositionBuffer;
}

/// <summary>
/// Called when the a tank overlaps with this Tank Ammunition Item.
/// 
/// Parent Comment: Event called when something starts to overlaps this component, for example a player walking into a trigger.
/// For events when objects have a blocking collision, for example a player hitting a wall, see 'Hit' events.
///	
/// @note Both this component and the other one must have GetGenerateOverlapEvents() set to true to generate overlap events.
/// @note When receiving an overlap from another object's movement, the directions of 'Hit.Normal' and 'Hit.ImpactNormal'
/// will be adjusted to indicate force from the other object against this object.
/// 
/// </summary>
/// <param name="overlappedComp">A pointer to the specific component in this Actor that was involved with the overlap.</param>
/// <param name="otherActor">A pointer to the Actor whose component triggered the overlap.</param>
/// <param name="otherComp">A pointer to the specific component on otherActor that caused the overlap</param>
/// <param name="otherBodyIndex">The index of the body part that was overlapped. This is relevant for skeletal meshes or other objects with multiple physics bodies, and it is usually 0 for simple components.</param>
/// <param name="bFromSweep">True if the overlap was the result of a sweep test, false otherwise.</param>
/// <param name="sweepResult">Struct which contains detailed information about the overlap, such as the location and normal of the impact point.</param>
void ATankAmmunitionItem::OnTankItemBeginOverlap(UPrimitiveComponent* overlappedComp, AActor* otherActor, UPrimitiveComponent* otherComp, int32 otherBodyIndex, bool bFromSweep, const FHitResult& sweepResult)
{
	Super::OnTankItemBeginOverlap(overlappedComp, otherActor, otherComp, otherBodyIndex, bFromSweep, sweepResult);

	FVector randPosition = FVector(FMath::FRandRange(myMinXPosition, myMaxXPosition), FMath::FRandRange(myMinYPosition, myMaxYPosition), 0.0f);
	float heightPosition = GetLandscapeHeightAtLocation(randPosition) + myRespawnDropHeight;

	SetActorLocation(FVector(randPosition.X, randPosition.Y, heightPosition));
}