/*
* Neural Tanks is a game created by Eddie O'Hagan that leverages Neural Networking
* to use for AI. This was created in Unreal Engine 4 which is developed by Epic Games.
* (Copyright Epic Games, Inc. All Rights Reserved.)
*
* Smart Tanks was originally developed by Mat Buckland in 2002 as an example
* on how Neural Networking can be used to train a set of minesweepers
* to collect mines. I (Eddie O'Hagan) updated this project in 2024 to use more
* Object Oriented Programming and modern practices. To see the original tutorial;
* visit <see href="http://www.ai-junkie.com/ann/evolved/nnt1.html">Neural Networking Tutorial</see>
* and <see href="http://www.ai-junkie.com/ga/intro/gat1.html">Genetic Algorithm Tutorial</see>
*/
#include "NeuralTanksAIController.h"
#include <BehaviorTree/BehaviorTree.h>
#include <BehaviorTree/BlackboardComponent.h>
#include "../Pawns/AITankPawn.h"
#include "../Utils/NeuralTanksUtils.h"
#include <NavigationSystem.h>

/// <summary>
/// Called when this controller starts posessing the tank pawn.
/// Starts the behavior tree for this AI.
/// </summary>
/// <param name="possessedPawn">The pawn that is being posessed.</param>
void ANeuralTanksAIController::OnPossess(APawn* possessedPawn)
{
	UBehaviorTree* theBehaviorTree;
	UBehaviorTreeComponent* theBehaviorTreeComponent;

	Super::OnPossess(possessedPawn);

	myAITankPawnOwner = Cast<AAITankPawn>(possessedPawn);
	if (myAITankPawnOwner != NULL)
	{
		theBehaviorTree = myAITankPawnOwner->GetBehaviorTree();
		if (theBehaviorTree != NULL)
		{
			theBehaviorTreeComponent = Cast<UBehaviorTreeComponent>(GetComponentByClass(UBehaviorTreeComponent::StaticClass()));
			theBehaviorTreeComponent->StartTree(*theBehaviorTree);
		}
		else
		{
			UNeuralTanksUtils::WriteToLog("ANeuralTanksAIController::OnPossess() Error: Behavior tree is missing for possessed pawn.");
		}
	}
	else
	{
		UNeuralTanksUtils::WriteToLog("ANeuralTanksAIController::OnPossess() Error: Possessed pawn was NOT an AITankPawn.");
	}
}
